#!/bin/sh
# Copyright (c) 2000-2012 Synology Inc. All rights reserved.

init_env() {
	PATH="/usr/bin:/bin"
	PATH_LOG="/var/log"

	SZD_PID="/var/run"
}
init_lsb_status() { # void
	# XXX these defined in /etc/rc.subr just include to reference

	# According to LSB 3.1 (ISO/IEC 23360:2006), the `status` init-scripts
	# action should return the following exit status codes.
	#
	LSB_STAT_RUNNING=0      # program is running or service is OK
	LSB_STAT_DEAD_FPID=1    # program is dead and /var/run pid file exists
	LSB_STAT_DEAD_FLOCK=2   # program is dead and /var/lock lock file exists
	LSB_STAT_NOT_RUNNING=3  # program is not runnning
	LSB_STAT_UNKNOWN=4      # program or service status is unknown
	# 5-99                  # reserved for future LSB use
	# 100-149               # reserved for distribution use
	# 150-199               # reserved for application use
	# 200-254               # reserved

	# Non-status init-scripts actions should return an exit status of zero if
	# the action was successful. Otherwise, the exit status shall be non-zero.
	#
	LSB_SUCCESS=0           # successful
	LSB_ERR_GENERIC=1       # generic or unspecified error
	LSB_ERR_ARGS=2          # invalid or excess argument(s)
	LSB_ERR_UNIMPLEMENTED=3 # unimplemented feature (for example, `reload`)
	LSB_ERR_PERM=4          # user had insufficient privilege
	LSB_ERR_INSTALLED=5     # program is not installed
	LSB_ERR_CONFIGURED=6    # program is not configured
	LSB_NOT_RUNNING=7       # program is not running
	# 8-99                  # reserved for future LSB use
	# 100-149               # reserved for distribution use
	# 150-199               # reserved for application use
	# 200-254               # reserved
}
source /etc/rc.subr || init_lsb_status

set_smbd_affinity() {
	uname -a | grep -i qoriq > /dev/null 2>&1
	local isQorIQ="$?"

	if [ 0 -eq ${isQorIQ} ]; then
		for each_pid in `pidof smbd`; do
			/usr/bin/taskset -p 2 $each_pid > /dev/null 2>&1
		done
	fi
}

# util functions
syslog() { # logger args
	local ret=$?
	logger -p user.err -t $(basename $0)\($$\) "$@"
	return $ret
}
message() { # echo args
	local ret=$?
	echo "$@"
	return $ret
} >&2
warn() { # echo args
	local ret=$?
	local H="[33m" E="[0m"
	echo -en "$H$@" ; echo "$E"
	return $ret
} >&2

is_alive() { # <pid> [proc name]
	local pid=${1:?"error params"}
	local procname="$2"

	if [ ! -r "/proc" -o -z "${procname:-}" ]; then
		kill -0 "$pid" 2>/dev/null
	else
		bin_file=`readlink "/proc/$pid/exe"`
		[ x"$(basename "$bin_file")" = x"$procname" ]
	fi
}
is_any_running() {
	local i=
	for i in "$@"; do
		if pidof $i >/dev/null; then
			return 0
		fi
	done
	return 1
}
wait_migrate() {
	local retry=0
	local printer_tdbs="ntprinters.tdb ntforms.tdb ntdrivers.tdb"
	#check smbd exist per 5s
	#Totally wait 120s
	while [ $retry -le 24 ]
	do
		retry=$(( $retry+1 ))
		if pidof smbd>/dev/null; then
			sleep 5s
			continue
		fi
		#check nmbd prevent S80samba stop
		sleep 1s
		if pidof nmbd>/dev/null; then
			break
		else
			return 0
		fi
	done
	#check complete move tdb files
	for i in $printer_tdbs; do
		if [ -f /var/run/${i}.bkp ]; then
			mv -f /var/run/${i}.bkp /var/run/${i}.bkp.bak
		fi
		if [ -f /var/run/${i} ]; then
			mv -f /var/run/${i} /var/run/${i}.bak
		fi
	done

	#check success
	if pidof smbd >/dev/null; then
		return 0
	fi

	#smbd start fail
	if [ -f /var/run/registry.tdb.bkp ]; then
		cp /var/run/registry.tdb /var/run/registry.tdb.bak
		cp /var/run/registry.tdb.bkp /var/run/registry.tdb
	fi

	#check nmbd exist; avoid S80samba.sh stop during the 2 min
	if ! pidof nmbd>/dev/null; then
		return 0
	fi
	if /usr/syno/sbin/smbd "$@"; then
		return 0
	fi
	return 1
}
check_migrate() {
	local printer_tdbs="ntprinters.tdb ntforms.tdb ntdrivers.tdb"
	local is_migrate=0
	local i=
	for i in $printer_tdbs; do
		if [ -f /var/run/${i}.bkp ]; then
			is_migrate=1
			break
		fi
	done
	if [ $is_migrate = 0 ]; then
		return 0
	fi
	/usr/syno/etc/rc.d/S80samba.sh wait_migrate "$@" &
	return 0
}
proc_ppid() {
	local pid=$1;
	local ppid="";
	if [ -z $pid ]; then
		pid=$$;
	fi
	if [ -f "/proc/$pid/status" ]; then
		echo `grep "PPid" /proc/$pid/status | cut -d: -f2`
	else
		return 1;
	fi
}
proc_info() { # <proc name> [pid file]
	local procname=${1:?"error params"}
	local pidfile=${2:-"$SZD_PID/$procname.pid"}

	local pid= running= i=
	if [ -r "$pidfile" ]; then
		pid=`cat "$pidfile"`
		if is_alive "$pid" "$procname"; then
			running="R"
		fi
	fi
	if pidof "$procname" >/dev/null; then
		running=${running:-"?"}
	else
		running="-"
	fi

	if [ ! "$_proc_info_header_dumped" ]; then
		printf "%-15s %-7s %4s  other pids\n" "procname" "pidfile" "stat"
		_proc_info_header_dumped=y
	fi

	printf "%-15s %-7s %4s  " "$procname" "$pid" "$running"
	for i in `pidof "$procname"`; do
		[ "$i" -eq "$pid" ] && continue

		echo -n "$i("`proc_ppid "$i"`") "
	done
	echo ""
}
proc_stop() {
	local procname=${1:?"error params"}
	local pidfile=${2:-"$SZD_PID/$procname.pid"}

	message -n "stop $procname ... "

	if ! pidof "$procname" >/dev/null; then
		message "not running"
		return ${LSB_NOT_RUNNING:-7}
	fi

	if [ -f "$pidfile" ]; then
		local pid=`cat "$pidfile"`
		if kill $pid; then
			message ""
		else
			warn "dead pid file"
			killall "$procname"
		fi
	else
		warn "no pid file, but process exist"
		killall "$procname"
	fi
}
proc_wait_stop() {
	local procname=${1:?"error params"}
	local retry=10
	local d1=`date +%s`

	while [ $retry -gt 0 ] && pidof "$procname" >/dev/null; do
		sleep 1
		retry=$((retry-1))
	done

	local d2=`date +%s`

	if pidof "$procname" >/dev/null; then
		killall -9 "$procname"
		warn "$procname still running, wait $((d2 - d1)) sec, force kill"
	else
		message "$procname stoped ($((d2 - d1)) sec)" 
	fi
}
proc_signal() { # <signal> <proc name>
	local sig=${1:?"error params"}
	local procname=${2:?"error params"}
	local pidfile=${3:-"$SZD_PID/$procname.pid"}

	if [ -f "$pidfile" ]; then
		local pid=`cat "$pidfile"`
		if ! kill -$sig $pid; then
			warn "dead pid file"
		fi
	fi
}
get_first_volume()
{
	local VPATH=`grep volume /proc/mounts 2>/dev/null | awk '{print $2}' | head -1`
	echo $VPATH
}
log_rotate() { # <logname> [size]
	local logname=${1:?"error params"}
	local size=${2:-$((5*1024*1024))}
	local maxsize=$((10*1024*1024))
	local ext=".old"

	if ! [ -e "$logname" ]; then
		return
	fi

	local logsize=$(stat -t "$logname" | cut -d\  -f2)

	if [ "$logsize" -gt "$maxsize" ]; then
		# overwrite old one
		syslog "log file too large, remove [$logname]"
		rm -f "$logname"
	elif [ "$logsize" -gt "$size" ]; then
		mv -f "$logname" "$logname$ext"
	fi
}

# lsb util functions
lsb_status() { # <proc name> [pid file]
	local procname=${1:?"error params"}
	local pidfile=${2:-"$SZD_PID/$procname.pid"}

	if [ -f "$pidfile" ]; then
		local pid=`cat "$pidfile"`
		if is_alive "$pid" "$procname"; then
			printf "%-15s %s\n" "$procname:" running
			return ${LSB_STAT_RUNNING:-0}
		else
			warn "dead pid file: $pidfile"
			printf "%-15s %s\n" "$procname:" stopped
			return ${LSB_STAT_DEAD_FPID:-1}
		fi
	fi

	printf "%-15s %s\n" "$procname:" stopped
	return ${LSB_STAT_NOT_RUNNING:-3}
}

# smb util functions
smb_is_enabled() { # <void>
	local toEnable=$(get_key_value /etc/synoinfo.conf "runsamba")
	[ "${toEnable:-no"}" = "yes" ]
}
smb_remove_temp_tdbs() { # <void>
	local winbinddcache=`readlink /var/run/winbindd_cache.tdb`

	if is_any_running smbd nmbd winbindd; then
		return 1
	fi

	message "remove temp tdbs"

	rm -f /tmp/brlock.tdb /tmp/connections.tdb /tmp/gencache.tdb \
		/tmp/login_cache.tdb /tmp/messages.tdb /tmp/netsamlogon_cache.tdb \
		"/var/run/printing/*.tdb" /tmp/sessionid.tdb /tmp/locking.tdb \
		/tmp/unexpected.tdb /tmp/deferred_open.tdb "$winbinddcache" \
		/var/run/winbindd_cache.tdb /var/run/winbindd_cache.tdb.bkp \
		/var/run/group_mapping.tdb
}
smb_check_tdb() { # <tdb file>
	local tdbfile=${1:?"error param"}
	local backup=/usr/syno/bin/tdbbackup

	message -n "check tdb: $tdbfile ... "

	if [ -f "$tdbfile" ]; then
		if $backup -v "$tdbfile" >/dev/null 2>&1; then
			# tdb is good make another backup
			mv -f "$tdbfile.bkp" "$tdbfile.bkp.old"
			if $backup -s ".bkp" "$tdbfile" >/dev/null 2>&1; then
				message "done"
				rm -f "$tdbfile.bkp.old"
			else
				warn "backup failed"
				mv -f "$tdbfile.bkp.old" "$tdbfile.bkp"
			fi
		elif [ -f "$tdbfile.bkp" ]; then
			warn "corrupt, restore"
			if ! $backup -v -s ".bkp" "$tdbfile" >/dev/null; then
				warn "restore failed, remove it"
				rm -f "$tdbfile"
			fi
		else
			warn "corrupt, remove it"
			rm -f "$tdbfile"
		fi
	elif [ -f "$tdbfile.bkp" ];then
		warn "lost, use backup tdb"
		if ! $backup -v -s ".bkp" "$tdbfile" > /dev/null; then
			warn "restore failed"
		fi
	else
		message "not exist"
	fi

	return 0
}

smb_start_nmbd() {
	message -n "start nmbd ... "

	log_rotate "$PATH_LOG"/log.nmbd

	if pidof nmbd >/dev/null; then
		warn "nmbd is running: `pidof nmbd`"
		return 1
	fi

	if /usr/syno/sbin/nmbd "$@"; then
		message "ok"
		return 0
	else
		warn "failed"
		return 1
	fi
}
smb_start_winbindd() {
	local log=
	message "start winbindd ... "

	for log in "$PATH_LOG"/log.winbind* "$PATH_LOG"/log.wb-*; do
		log_rotate "$log"
	done

	if pidof winbindd >/dev/null; then
		warn "winbindd is running: `pidof winbindd`"
		return 1
	fi
	
	smb_check_tdb /var/run/winbindd_idmap.tdb

	local szd_volume=`get_first_volume`
	if [ "x$szd_volume" != "x" ]; then
		local winbindcache="$szd_volume/winbindd_cache.tdb"

		touch "$winbindcache"
		ln -s "$winbindcache" /var/run/winbindd_cache.tdb
	fi

	if /usr/syno/sbin/winbindd "$@"; then
		message "done"
		return 0
	else
		warn "failed"
		return 1
	fi
}
smb_start_smbd() {
	local smbspool=/var/spool/samba i=
	message "start smbd ... "

	log_rotate "$PATH_LOG"/log.smbd

	if pidof smbd >/dev/null; then
		warn "smbd is running: `pidof smbd`"
		return 1
	fi

	# FIXME remove smbspool
	[ -d "$smbspool" ] && rm -f "$smbspool/*"

	# check private dir
	smb_check_tdb /usr/syno/etc/private/secrets.tdb

	# check smb tdb
	for i in account_policy.tdb share_info.tdb registry.tdb; do
		smb_check_tdb /var/run/$i
	done

	if /usr/syno/sbin/smbd "$@"; then
		message "done"
		check_migrate "$@"

		{
			/bin/sleep 1
			set_smbd_affinity
		} &

		return 0
	else
		warn "failed"
		return 1
	fi
}

# actions
usage() { # void
	local H="[1m"
	local E="[0m"
	cat <<EOF
Usage: `basename $0` <actions>
Actions:
 $H start$E [options]            start samba by runkey in synoinfo.conf with
                             options.
 $H stop$E                       stop samba daemons
 $H restart$E                    restart service. equal to stop && start
 $H reload,hup$E                 reload config smb.conf
 $H force-reload$E               
 $H status$E                     show running status for samba daemon
 $H info$E                       show detail informations
 $H -h,--help,usage$E            show this help message
EOF
}
start() { # [options]
	local opt OPTARG OPTIND smb_opts="-D $@"
	if ! smb_is_enabled; then
		warn "samba is not configured for running"
		return ${LSB_ERR_CONFIGURED:-6}
	fi

	# FIXME workaround for samba 3.6.x
	if ! [ -d /usr/syno/var ]; then
		mkdir -p /usr/syno/var
	fi

	smb_remove_temp_tdbs
	
	smb_start_nmbd $smb_opts

	local security=`get_key_value /usr/syno/etc/smb.conf security`
	case $security in
		domain|ads) smb_start_winbindd $smb_opts;;
	esac

	smb_start_smbd $smb_opts
}
stop() { # [options]
	local smb_running nmbd_running winbindd_running

	proc_stop smbd
	if [ $? -ne ${LSB_NOT_RUNNING:-7} ]; then
		proc_wait_stop smbd &
	fi

	proc_stop winbindd
	if [ $? -ne ${LSB_NOT_RUNNING:-7} ]; then
		proc_wait_stop winbindd &
	fi

	proc_stop nmbd
	if [ $? -ne ${LSB_NOT_RUNNING:-7} ]; then
		proc_wait_stop nmbd &
	fi

	wait
	smb_remove_temp_tdbs
}
status() { # void
	lsb_status nmbd
	lsb_status winbindd
	lsb_status smbd
	local ret=$?
	return $ret
}
restart() { # <void>
	stop
	start "$@"
}
reload() {
	proc_signal hup nmbd
	proc_signal hup winbindd
	proc_signal hup smbd
}

info() { # <void>
	if mount | grep -q "smb"; then
		echo mount
	else
		echo not mount
	fi
	echo ====== proc info ======
	proc_info smbd
	proc_info nmbd
	proc_info winbindd

	echo ====== smbstatus ======
	/usr/syno/bin/smbstatus -d 0

	echo ====== tdbs ======
	echo `find /tmp /var/run /usr/syno/etc/ -name "*.tdb"`

	#file_dump /etc/resolv.conf
	#file_dump /usr/syno/etc/smb.conf "workgroup"
	#file_dump /var/log/winlock.state
	#file_dump /usr/syno/etc/private/workgroup
}

init_env
if [ $# -eq 0 ]; then
	action=status
else
	action=$1
	shift
fi

# dispatch actions
case $action in
	start|stop|status|usage|restart|reload)
		$action "$@"
		;;
	hup)
		reload "$@"
		;;
	force-reload)
		exit ${LSB_ERR_UNIMPLEMENTED:-3}
		;;
	-h|--help)
		usage "$@"
		;;
# for debugging
	info)
		$action "$@"
		;;
	test_*)
		local fn=${action#test_}
		$fn "$@"
		;;
	wait_migrate)
		wait_migrate "$@"
		;;
	*)
		usage "$@" >&2
		exit ${LSB_ERR_ARGS:-2}
		;;
esac
