#!/bin/sh
# Copyright (c) 2000-2006 Synology Inc. All rights reserved.

Httpd="/usr/syno/apache/bin/httpd"
HttpdConf="/usr/syno/apache/conf/httpd.conf-sys"
HttpdConfDefault="/usr/syno/apache/.conf.default/httpd.conf-sys"
HttpdSSLConf="/usr/syno/apache/conf/extra/httpd-ssl.conf-sys"
HttpdSSLConfCommon="/usr/syno/apache/conf/extra/httpd-ssl.conf-common"
MissingPHP="/usr/syno/synoman/phpsrc/web/sample.php"
PIDFILE="/var/run/httpd-sys.pid"
SSLKEYGEN="/usr/syno/etc.defaults/rc.ssl_keygen"
CGIKEYGEN="/usr/syno/etc.defaults/rc.cgi_keygen"
SitesAvailableDir="/usr/syno/etc/sites-available"
SitesEnabledDir="/usr/syno/etc/sites-enabled"
INTERCRT="/usr/syno/etc/ssl/ssl.intercrt/server-ca.crt"
AltPortSSLSettingConf="/usr/syno/apache/conf/extra/httpd-alt-port-ssl-setting.conf"

RunHTTPS=`/bin/get_key_value /etc/synoinfo.conf runHTTPS`

. /etc.defaults/rc.subr

CheckAltPortSslSetting()
{
	if [ -e $INTERCRT ]; then
		sed -e "/^#SSLCertificateChainFile/c\\SSLCertificateChainFile /usr/syno/etc/ssl/ssl.intercrt/server-ca.crt" \
			${AltPortSSLSettingConf} > ${AltPortSSLSettingConf}.$$
	else
		sed -e "/^SSLCertificateChainFile/c\\#SSLCertificateChainFile /usr/syno/etc/ssl/ssl.intercrt/server-ca.crt" \
			${AltPortSSLSettingConf} > ${AltPortSSLSettingConf}.$$
	fi

	if [ -s "${AltPortSSLSettingConf}.$$" ]; then
		mv ${AltPortSSLSettingConf}.$$ ${AltPortSSLSettingConf}
	fi

	rm -f ${AltPortSSLSettingConf}.$$ 2>/dev/null
}

GenerateConf()
{
	ADMIN_PORT=`/bin/get_key_value /etc/synoinfo.conf admin_port`
	if [ -z "$ADMIN_PORT" ]; then
		ADMIN_PORT=`/bin/get_key_value /etc.defaults/synoinfo.conf admin_port`
		if [ -z "$ADMIN_PORT" ]; then
			ADMIN_PORT="5000"
		fi
	fi
	ADMIN_SSL_PORT=`/bin/get_key_value /etc/synoinfo.conf secure_admin_port`
	if [ -z "$ADMIN_SSL_PORT" ]; then
		ADMIN_SSL_PORT=`/bin/get_key_value /etc.defaults/synoinfo.conf secure_admin_port`
		if [ -z "$ADMIN_SSL_PORT" ]; then
			ADMIN_SSL_PORT="5001"
		fi
	fi

	cp -f ${HttpdConf} ${HttpdConf}.$$
	if [ $? -ne 0 ]; then
		logger -p user.err -t `basename $0` "backup ${HttpdConf} failed."
		rm -f ${HttpdConf}.$$
		return 1
	fi
	sed -e "/^Listen/c\\Listen ${ADMIN_PORT}" \
	    -e "/^ServerName/c\\ServerName *:${ADMIN_PORT}" \
	    -e "/redirect.cgi?/c\\	RewriteRule ^/(.*)$ /usr/syno/synoman/redirect.cgi?${ADMIN_PORT}" \
	    ${HttpdConf} > ${HttpdConf}.$$
	if [ -s "${HttpdConf}.$$" ]; then
		mv ${HttpdConf}.$$ ${HttpdConf}
	fi

	cp -f ${HttpdSSLConf} ${HttpdSSLConf}.$$
	if [ $? -ne 0 ]; then
		logger -p user.err -t `basename $0` "backup ${HttpdSSLConf} failed."
		rm -f ${HttpdSSLConf}.$$
		return 1
	fi
	
	sed -e "/^Listen/c\\Listen ${ADMIN_SSL_PORT}" \
	    -e "/^<VirtualHost/c\\<VirtualHost *:${ADMIN_SSL_PORT}>" \
	    -e "/redirect.cgi?/c\\	RewriteRule ^/(.*)$ /usr/syno/synoman/redirect.cgi?${ADMIN_SSL_PORT}" \
	    ${HttpdSSLConf} > ${HttpdSSLConf}.$$
	if [ -s "${HttpdSSLConf}.$$" ]; then
		mv ${HttpdSSLConf}.$$ ${HttpdSSLConf}
	fi
	rm -f ${HttpdConf}.$$ ${HttpdSSLConf}.$$ 2>/dev/null

	if [ -s ${HttpdSSLConfCommon} ]; then
		if [ -f $INTERCRT ]; then
			sed -e "/^#SSLCertificateChainFile/c\\SSLCertificateChainFile /usr/syno/etc/ssl/ssl.intercrt/server-ca.crt" \
			${HttpdSSLConfCommon} > ${HttpdSSLConfCommon}.$$
		else
			sed -e "/^SSLCertificateChainFile/c\\#SSLCertificateChainFile /usr/syno/etc/ssl/ssl.intercrt/server-ca.crt" \
			${HttpdSSLConfCommon} > ${HttpdSSLConfCommon}.$$
		fi

		if [ -s "${HttpdSSLConfCommon}.$$" ]; then
			mv ${HttpdSSLConfCommon}.$$ ${HttpdSSLConfCommon}
		fi

		rm -f ${HttpdSSLConfCommon}.$$ 2>/dev/null
	fi

	if [ -f "${MissingPHP}" ]; then
		cp -f ${MissingPHP} ${MissingPHP}.$$
		if [ $? -ne 0 ]; then
			logger -p user.err -t `basename $0` "backup ${MissingPHP} failed."
			rm -f ${MissingPHP}.$$
			return 1
		fi
		sed "/^\$ADMIN_PORT/c\\\$ADMIN_PORT = \"${ADMIN_PORT}\";" ${MissingPHP} > ${MissingPHP}.$$
		if [ -s "${MissingPHP}.$$" ]; then
			mv ${MissingPHP}.$$ ${MissingPHP}
		else
			rm -f ${MissingPHP}.$$ 2>/dev/null
		fi
	fi
}

MigrateAltPortConfByPath()
{
	local OldConf="$1"
	local DsmAppName="$2"
	local Template="$3"
	local NewConfName="$4"
	local PortKey="$5"

	if [ ! -f "$OldConf" ]; then
		return 1
	fi

	local Port=`/bin/sed -n "s,Listen \([0-9]\+\),\1,p" $OldConf  2>/dev/null`

	if [ -n "$Port" ]; then
		mv "$OldConf" "$OldConf.bak"

		cp -f $Template $SitesAvailableDir/$NewConfName

		/bin/sed -i "s,{$PortKey},$Port,g"  $SitesAvailableDir/$NewConfName
		/bin/sed -i "s,{DSM_APP_NAME},$DsmAppName,g"  $SitesAvailableDir/$NewConfName

		ln -sf  $SitesAvailableDir/$NewConfName  $SitesEnabledDir/$NewConfName
	fi
}

MigrateAltPortConfHttp()
{
	local OldConf="$1"
	local DsmAppName="$2"
	local Template="/usr/syno/apache/conf/extra/httpd-template-alt-port.conf"
	local NewConfName="$DsmAppName.alt_port.conf"
	local PortKey="HTTP_PORT"

	MigrateAltPortConfByPath "$OldConf" "$DsmAppName" "$Template" "$NewConfName" "$PortKey"
}

MigrateAltPortConfSsl()
{
	local OldConf="$1"
	local DsmAppName="$2"
	local Template="/usr/syno/apache/conf/extra/httpd-template-alt-port-ssl.conf"
	local NewConfName="$DsmAppName.alt_port_ssl.conf"
	local PortKey="HTTPS_PORT"

	MigrateAltPortConfByPath "$OldConf" "$DsmAppName" "$Template" "$NewConfName" "$PortKey"
}

MigrateAltPortConf()
{
	local FileStationConf="/usr/syno/etc/httpd-filestation.conf-sys"
	local FileStationSslConf="/usr/syno/etc/httpd-ssl-filestation.conf-sys"
	local FileStationDsmAppName="SYNO.SDS.App.FileStation3.Instance"
	local SurveillanceStationConf="/usr/syno/etc/httpd-surveillancestation.conf-sys"
	local SurveillanceStationSslConf="/usr/syno/etc/httpd-ssl-surveillancestation.conf-sys"
	local SurveillanceStationDsmAppName="SYNO.SDS.SurveillanceStation"

	# migrate File Station conf
	MigrateAltPortConfHttp $FileStationConf $FileStationDsmAppName
	MigrateAltPortConfSsl $FileStationSslConf $FileStationDsmAppName

	# migrate Surveillance Station conf
	MigrateAltPortConfHttp $SurveillanceStationConf $SurveillanceStationDsmAppName
	MigrateAltPortConfSsl $SurveillanceStationSslConf $SurveillanceStationDsmAppName
}

CheckSitesEnabledConf()
{
	# check all symbolic link not broken
	local files=`/usr/bin/find "$SitesEnabledDir" -name "*.conf" 2>/dev/null`
	for f in $files
	do
		if [ ! -r "$f" ]; then
			rm "$f"
		fi
	done
}

RemoveSitesEnabledConf()
{
	# remove all symbolic link
	local files=`/usr/bin/find "$SitesEnabledDir" -name "*.conf" 2>/dev/null`
	for f in $files
	do
		rm "$f"
	done
}

CheckPID() {
	# check for pidfile
	if [ -f $PIDFILE ] ; then
		PID=`cat $PIDFILE`
		if [ "x$PID" != "x" ] && kill -0 $PID 2>/dev/null ; then
			STATUS="httpd (pid $PID) running"
			RUNNING=1
		else
			STATUS="httpd (pid $PID?) not running"
			RUNNING=0
		fi
	else
		STATUS="httpd (no pid file) not running"
		RUNNING=0
	fi
}

BIN_RELAYD="/usr/syno/sbin/synorelayd"
UpdateRelayd()
{
	local relayd_arg=

	/usr/syno/etc/rc.d/S13synorelayd.sh status
	if [ $? -ne $LSB_STAT_RUNNING ] ; then
	    return
	fi

	case "$RunHTTPS" in
		"no")
		relayd_arg="$relayd_arg --remove dms_https --remove audio_https"
		;;
		"yes")
		if $BIN_RELAYD -S | grep '"audio_http"' >/dev/null; then
			relayd_arg="$relayd_arg --add audio_https"
		fi
		;;
	esac

	if [ -z "$relayd_arg" ]; then
		relayd_arg="--hup"
	fi

	$BIN_RELAYD $relayd_arg
	return 0
	# ignore relayd return value.
	# when relayd is not running, hup will return false
}

CheckPID

case $1 in
[Ss][Tt][Oo][Pp])
	if [ $RUNNING -eq 0 ]; then
		echo $STATUS
	else
		APACHE_SYS_PID=$PID

		if kill $PID ; then
			echo "$0: system httpd stopped"
		else
			echo "$0: system httpd could not be stopped"
			exit 1;
		fi

		MAXTRY=30
		i=0
		while [ $i -lt $MAXTRY -a $RUNNING -eq 1 ] ; do
			CheckPID
			if [ $RUNNING -eq 1 ] ; then
				sleep 1
			fi
			i=`expr $i + 1`
		done

		kill -9 $APACHE_SYS_PID 2> /dev/null;
	fi

        UpdateRelayd
	;;
[Ss][Tt][Aa][Rr][Tt])
	$SSLKEYGEN > /dev/null 2>&1
	$CGIKEYGEN > /dev/null 2>&1 &
	if [ ! -x "$Httpd" ]; then
		echo "$Httpd is not executable."
		exit 1;
	fi

	if [ $RUNNING -eq 1 ]; then
		echo "$0 $ARG: httpd (pid $PID) already running"
	else
		GenerateConf
		MigrateAltPortConf
		CheckAltPortSslSetting
		CheckSitesEnabledConf

		startArg=""
		if [ "yes" = "$RunHTTPS" ]; then
			startArg="$startArg -DSSL"
		fi

		startArg="$startArg -f ${HttpdConf}"

		echo "Start System Apache Server ..... $startArg"
		if $Httpd $startArg; then
			echo "$0: system httpd started"
		else
			RemoveSitesEnabledConf
			if $Httpd $startArg; then
				echo "$0: system httpd started"
			else
				echo "Recover to default setting"
				if $Httpd -f ${HttpdConfDefault}; then
					echo "$0: system httpd started with default setting"
					logger -p user.err -t `basename $0` "system httpd started with default setting"
					synodsmnotify @administrators dsmnotify:system_event widget:apache_statup_failed
				else
					echo "$0: system httpd could not be started"
					logger -p user.err -t `basename $0` "system httpd could not be started"
				fi
				exit 1;
			fi
		fi
        UpdateRelayd
	fi
	;;
[Rr][Ee][Ss][Tt][Aa][Rr][Tt])
	if [ $RUNNING -eq 0 ]; then
		$0 start
	else
		$0 stop
		sleep 3
		$0 start
	fi
	;;
[Ss][Tt][Aa][Tt][Uu][Ss])
	if [ $RUNNING -eq 0 ]; then
		return $LSB_STAT_NOT_RUNNING
	else
		return $LSB_STAT_RUNNING
	fi
	;;
isDefault)
	ps -Iww | grep ${Httpd} | grep ${HttpdConfDefault} >/dev/null 2>&1
	if [ $? -eq 0 ]; then
		echo "Is Default mode"
		exit 0;
	else
		echo "Is not Default mode"
		exit 1;
	fi
	;;
*)
	echo "Usage: $0 [start|stop|restart|status]"
	;;
esac
exit 0;
