#!/bin/sh

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/syno/sbin:/usr/syno/bin

KERNEL_VERSION=`uname -r`
CONF="/etc/synoinfo.conf"
DEFAULT_CONF="/etc.defaults/synoinfo.conf"
V4ONLY=`get_key_value $DEFAULT_CONF ipv4only`
SZD_FW_SECURITY_ROOT="/etc/fw_security"
SZD_FW_SECURITY_SYS_CONF="$SZD_FW_SECURITY_ROOT/sysconf"
SZK_FW_S_DOS_ENABLE="dos_protect_enable"
SZK_FW_S_PPTP_ENABLE="pptp_passthrough_enable"
SZK_FW_S_IPSEC_ENABLE="ipsec_passthrough_enable"
SZK_FW_S_L2TP_ENABLE="l2tp_passthrough_enable"
SZF_FW_S_DOS_PATTERN="dos.pattern"
SZF_FW_S_PPTP_PATTERN="pptp_passthrough.pattern"
SZF_FW_S_IPSEC_PATTERN="ipsec_passthrough.pattern"
SZF_FW_S_L2TP_PATTERN="l2tp_passthrough.pattern"
SZF_RULES_SECURITY_DUMP="/etc/firewall_rules_security.dump"
SZF_RULES_6_SECURITY_DUMP="/etc/firewall_6_rules_security.dump"
SZK_SUPPORT_NET_TOPOLOGY="support_net_topology"
SZK_SUPPORT_PCI_WIFI="support_pci_wifi"
SZK_NET_TOPOLOGY="net_topology"
ROUTER="router"
SERV_DOS="fw_dos"
SERV_VPN_PASSTHROUGH="fw_vpn_passthrough"
PPPOE="pppoe"
WLAN0="wlan0"
INSMOD_DOS=0


case "$KERNEL_VERSION" in
"2.6.3"[2-6]*)
	KERNEL_FIREWALL="x_tables.ko ip_tables.ko iptable_filter.ko nf_conntrack.ko nf_defrag_ipv4.ko nf_conntrack_ipv4.ko ipt_LOG.ko xt_multiport.ko xt_tcpudp.ko xt_state.ko"
	IPV6_MODULES="ip6_tables.ko ip6table_filter.ko nf_conntrack_ipv6.ko ip6t_LOG.ko"
	DOS_MODULES="${KERNEL_FIREWALL} xt_limit.ko"
	VPN_PASSTHROUGH_MODULES="${KERNEL_FIREWALL} nf_conntrack_proto_gre.ko nf_nat_proto_gre.ko nf_conntrack_pptp.ko nf_nat_pptp.ko"
	;;
"2.6.3"[7-9]*|"3."*)
	KERNEL_FIREWALL="x_tables.ko ip_tables.ko iptable_filter.ko nf_conntrack.ko nf_defrag_ipv4.ko nf_conntrack_ipv4.ko ipt_LOG.ko xt_multiport.ko xt_tcpudp.ko xt_state.ko"
	IPV6_MODULES="ip6_tables.ko ip6table_filter.ko nf_defrag_ipv6.ko nf_conntrack_ipv6.ko ip6t_LOG.ko"
	DOS_MODULES="${KERNEL_FIREWALL} xt_limit.ko"
	VPN_PASSTHROUGH_MODULES="${KERNEL_FIREWALL} nf_conntrack_proto_gre.ko nf_nat_proto_gre.ko nf_conntrack_pptp.ko nf_nat_pptp.ko"
	;;
*)
	echo "******iptables: Kernel version not supported******"
	;;
esac

reverse_syno() {
	local modules=$1
	local mod
	local ret=""

	for mod in $modules; do
	    ret="$mod $ret"
	done

	echo $ret
}

check_vpn_passthrough() {
	local ret=0
	local support_net_topology=`get_key_value $DEFAULT_CONF $SZK_SUPPORT_NET_TOPOLOGY`
	local support_pci_wifi=`get_key_value $DEFAULT_CONF $SZK_SUPPORT_PCI_WIFI`
	local topology=`get_key_value $CONF $SZK_NET_TOPOLOGY`

	if [ "yes" = "${support_net_topology}" -a "yes" = "${support_pci_wifi}" -a "${ROUTER}" = "${topology}" ]; then
		ret=1
	fi
	return $ret
}

#ipv6 ip6tables
ipv6_rule_apply() {
	while read line
	do
		ip6tables $line
	done < $1
}

#ipv4 iptables, ipv4 modules must been loaded, because of following ipv6 firewall may use those modules
ipv4_rule_apply() {
	while read line
	do
		iptables $line
	done < $1
}

modules_reload() {
	local modules_reverse=""

	##### remove modules #####
	#if old settings have dos/vpn passthrough but not for new settings, we need to clean old settings related modules
	if [ 0 -eq $support_vpn_passthrough ]; then
		modules_reverse=`reverse_syno "$VPN_PASSTHROUGH_MODULES"`
		/usr/syno/bin/iptablestool --rmmod $SERV_VPN_PASSTHROUGH $modules_reverse
	fi
	if [ 0 -eq $INSMOD_DOS ]; then
		#before remove kernel modules, DOS_PROTECT chain should be delete first, or it will fail to remove iptable_filter.ko
		if [ "yes" != "${V4ONLY}" ]; then
			ret=`/bin/grep filter /proc/net/ip6_tables_names 2>&1`
			if [ 0 -eq $? -a "xfilter" = "x$ret" ]; then
				/sbin/ip6tables -X DOS_PROTECT
			fi
			modules_reverse=`reverse_syno "$IPV6_MODULES"`
			/usr/syno/bin/iptablestool --rmmod $SERV_DOS $modules_reverse
		fi
		ret=`/bin/grep filter /proc/net/ip_tables_names 2>&1`
		if [ 0 -eq $? -a "xfilter" = "x$ret" ]; then
			/sbin/iptables -X DOS_PROTECT
		fi
		modules_reverse=`reverse_syno "$DOS_MODULES"`
		/usr/syno/bin/iptablestool --rmmod $SERV_DOS $modules_reverse
	fi

	##### insert modules #####
	#if new settings have dos/vpn passthrough, we need to insmod the related modules
	if [ 1 -eq $INSMOD_DOS ]; then
		/usr/syno/bin/iptablestool --insmod $SERV_DOS ${DOS_MODULES}
		if [ "yes" != "${V4ONLY}" ]; then
			/usr/syno/bin/iptablestool --insmod $SERV_DOS ${IPV6_MODULES}
		fi
	fi
	#load VPN passthrough related modules only for 213air router mode
	if [ 1 -eq $support_vpn_passthrough ]; then
		/usr/syno/bin/iptablestool --insmod $SERV_VPN_PASSTHROUGH ${VPN_PASSTHROUGH_MODULES}
	fi
}

dump_rules_clear() {
	for i in $@;
	do
		if [ -e $i ]; then
			rm -f $i &> /dev/null
		fi
	done
}

start() {
	local has_portfw_rule=0

	if [ 1 -eq $1 ]; then
		has_portfw_rule=1
	fi

	fw_security_dump $has_portfw_rule

	modules_reload

	[ -f $SZF_RULES_SECURITY_DUMP ] && ipv4_rule_apply $SZF_RULES_SECURITY_DUMP
	[ "yes" != "${V4ONLY}" -a -f $SZF_RULES_6_SECURITY_DUMP ] && ipv6_rule_apply $SZF_RULES_6_SECURITY_DUMP

	return 0
}

ipv6_stop() {
	local modules_reverse=""

	echo ""
	echo "Unloading kernel ipv6 netfilter modules... "
	modules_reverse=`reverse_syno "$IPV6_MODULES"`
	/usr/syno/bin/iptablestool --rmmod $SERV_DOS $modules_reverse

	dump_rules_clear $SZF_RULES_6_SECURITY $SZF_RULES_6_SECURITY_DEL
}

ipv4_stop() {
	local modules_reverse=""

	echo ""
	echo "Unloading kernel ipv4 netfilter modules... "
	modules_reverse=`reverse_syno "$VPN_PASSTHROUGH_MODULES"`
	/usr/syno/bin/iptablestool --rmmod $SERV_VPN_PASSTHROUGH $modules_reverse
	modules_reverse=`reverse_syno "$DOS_MODULES"`
	/usr/syno/bin/iptablestool --rmmod $SERV_DOS $modules_reverse

	dump_rules_clear $SZF_RULES_SECURITY $SZF_RULES_SECURITY_DEL
}

stop() {
	[ "yes" != "${V4ONLY}" ] && ipv6_stop
	ipv4_stop

	return 0
}

restart() {
	stop
	start $1
}

get_interface() {
	local out_if=""
	local interface=""

	interface=`/usr/syno/sbin/synonet --show | grep "interface: " | awk '{print $NF}'`
	if [ $? -eq 0 ]; then
		out_if=$interface
	fi

	#get wlan0
	support_wifi=`/bin/get_key_value $DEFAULT_CONF support_wireless`
	if [ "xyes" = "x$support_wifi" ]; then
		tmp=`/bin/echo $out_if | grep $WLAN0`
		if [ 0 -ne $? ]; then
			out_if="$out_if $WLAN0"
		fi
	fi

	#get pppoe
	out_if="$out_if $PPPOE"

	echo $out_if
}

fw_security_dump() {
	local ret=""
	local interface=""
	local all_if=""
	local modconf=""
	local ifName=""
	local has_portfw_rule=$1

	dump_rules_clear $SZF_RULES_SECURITY_DUMP $SZF_RULES_6_SECURITY_DUMP

	for modfile in $SZD_FW_SECURITY_ROOT/*.conf.tmp ;
	do
		modconf=`basename $modfile | sed 's/.conf.tmp$//g'`
	done

	all_if=`get_interface`

	for inf in $all_if ;
	do
		if [ "$modconf" = "$inf" ]; then
			file=$modfile
		else
			file=$SZD_FW_SECURITY_ROOT/$inf.conf
		fi
		interface=$inf
		if [ "$PPPOE" = "$inf" ]; then
			ret=`/usr/syno/sbin/synonet --check_pppoe`
			if [ 0 -ne $? ]; then
				continue
			fi
			ifName=`/usr/syno/sbin/synonet --get_if pppoe`
			if [ 0 -ne $? ]; then
				continue
			fi
			interface=$ifName
		fi

		ret=`/bin/get_key_value $file $SZK_FW_S_DOS_ENABLE`
		if [ "xyes" = "x$ret" ]; then
			INSMOD_DOS=1
			cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_DOS_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_SECURITY_DUMP
			[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_DOS_PATTERN | sed -e s/INTERFACE/"$interface"/g | grep -v icmp >> $SZF_RULES_6_SECURITY_DUMP
		fi

		#dump vpn passthrough rules only when (has portfw & router mode) -> it implys iptables has nat table
		if [ 1 -eq $has_portfw_rule -a 1 -eq $support_vpn_passthrough ]; then
			if [ "eth0" = "$interface" -o "wlan0" = "$interface" ]; then
				continue
			fi
			ret=`/bin/get_key_value $file $SZK_FW_S_PPTP_ENABLE`
			if [ "xyes" = "x$ret" ]; then
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_PPTP_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_PPTP_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_6_SECURITY_DUMP
			else
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_PPTP_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_PPTP_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_6_SECURITY_DUMP
			fi

			ret=`/bin/get_key_value $file $SZK_FW_S_IPSEC_ENABLE`
			if [ "xyes" = "x$ret" ]; then
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_IPSEC_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_IPSEC_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_6_SECURITY_DUMP
			else
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_IPSEC_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_IPSEC_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_6_SECURITY_DUMP
			fi

			ret=`/bin/get_key_value $file $SZK_FW_S_L2TP_ENABLE`
			if [ "xyes" = "x$ret" ]; then
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_L2TP_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_L2TP_PATTERN | sed -e s/INTERFACE/"$interface"/g >> $SZF_RULES_6_SECURITY_DUMP
			else
				cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_L2TP_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_SECURITY_DUMP
				[ "yes" != "${V4ONLY}" ] && cat $SZD_FW_SECURITY_SYS_CONF/$SZF_FW_S_L2TP_PATTERN | sed -e s/INTERFACE/"$interface"/g | sed -e s/ACCEPT/DROP/g >> $SZF_RULES_6_SECURITY_DUMP
			fi
		fi
	done

	return 0;
}

# do operations if support_fw_security="yes", or just return.
support_fw_security=`/bin/get_key_value $DEFAULT_CONF support_fw_security`
if [ "xyes" != "x$support_fw_security" ]; then
	echo "This platform doesn't support firewall security setting!"
	exit 1;
fi

check_vpn_passthrough
support_vpn_passthrough=$?

case "$1" in
	start)
		start $2
		RETVAL=$?
		;;
	force-reload)
		start $2
		RETVAL=$?
		;;
	stop)
		stop
		RETVAL=$?
		;;
	restart)
		restart $2
		RETVAL=$?
		;;
	*)
		echo $"Usage: ${0##*/} {start|stop|restart|force-reload}"
		RETVAL=2
		;;
esac

exit $RETVAL
