#!/bin/sh
# Copyright (c) 2000-2010 Synology Inc. All rights reserved.

IET_PID="/var/run/iscsi_trgt.pid"
IET_PROC="/proc/net/iet/volume"

ISCSITRGT_LOCK="/tmp/S78iscsitrg.lock"
ISCSITRASH_LOCK="/tmp/S78iscsitrash.lock"
ISCSITRGT_SUPPORT=`/bin/get_key_value /etc.defaults/synoinfo.conf support_iscsi_target`

LIO_CFS_DIR="/sys/kernel/config/target/iscsi"
TCM_CFS_DIR="/sys/kernel/config/target/core"

. /etc.defaults/rc.subr

KERNEL_VCODE=`KernelVersionCode $(KernelVersion)`
if [ $KERNEL_VCODE -ge $(KernelVersionCode "3") ]; then
	ISCSITRGT_TYPE=`/bin/get_key_value /etc.defaults/synoinfo.conf iscsi_target_type`
else
	ISCSITRGT_TYPE=`/bin/get_key_value /etc/synoinfo.conf iscsi_target_type`
	if [ "xiet" != "x${ISCSITRGT_TYPE}" -a "xlio" != "x${ISCSITRGT_TYPE}" ]; then
		ISCSITRGT_TYPE=`/bin/get_key_value /etc.defaults/synoinfo.conf iscsi_target_type`
	fi
fi

## IET - iSCSI Enterprise Target
iet_status()
{
	local success_cnt=0
	#check module statement number: 2 (iscsi_trgt, ietd)
	check_module_cnt=2

	RCMsg "Checking iSCSI Enterprise Target" \
	[ "x`/sbin/lsmod | /bin/grep iscsi_trgt | /usr/bin/awk '{ if ($1 == "iscsi_trgt") print $1 }'`" != "x" ]
	if [ $? -eq 0 ]; then
		success_cnt=$(($success_cnt + 1))
	fi

	RCMsg "Checking iSCSI Enterprise Target Daemon" \
	[ "x`/bin/ps | /bin/grep ietd | /bin/grep -v grep | /usr/bin/awk '{ if ($5 == "/usr/sbin/ietd") print $5 }'`" != "x" ]
	if [ $? -eq 0 ]; then
		success_cnt=$(($success_cnt + 1))
	fi

	return $success_cnt
}

iet_insmod()
{
	if [ -e $IET_PROC ]; then
		MsgFail "iSCSI kernel modules has been loaded... "
	else 
		/sbin/insmod /lib/modules/libcrc32c.ko 1>/dev/null 2>&1
		/sbin/insmod /lib/modules/crc32c.ko 1>/dev/null 2>&1
		/sbin/insmod /lib/modules/iscsi_trgt.ko 1>/dev/null 2>&1
		/usr/sbin/ietd 1>/dev/null 2>&1
		RCMsg "Loading iSCSI kernel modules" [ $? -eq 0 ]
	fi
}

iet_start()
{
	local enabled_count=`/usr/syno/bin/iscsitrgtool -enabled_count`
	if [ $enabled_count -eq 0 ]; then
		RCMsg "No iSCSI targets enabled, skip it."
		return $LSB_SUCCESS
	fi

	iet_insmod

	/usr/syno/bin/iscsitrgtool -startall > /dev/null 2>&1

	return $LSB_SUCCESS
}

iet_stop()
{
	# Removing iSCSI Target Devices
	/usr/sbin/ietadm --op delete > /dev/null 2>&1 # ietadm does not always provide correct exit values

	RCMsg "Stopping iSCSI Target Daemon" /usr/bin/killall ietd
	/bin/rm -f $IET_PID 2> /dev/null # pid file is not removed by iet
	sleep 1

	# Removing iSCSI Kernel Modules
	/sbin/rmmod iscsi_trgt 2>/dev/null
	/sbin/rmmod crc32c 2>/dev/null
	/sbin/rmmod libcrc32c 2>/dev/null

	return $LSB_SUCCESS
}

## LIO - Linux-iSCSI.org
lio_status()
{
	local success_cnt=0
	#check module statement number: 3 (configfs, target_core_mod, iscsi_target_mod)
	check_module_cnt=3

	RCMsg "Checking ConfigFS" \
	[ -d /sys/kernel/config ]
	if [ $? -eq 0 ]; then
		success_cnt=$(($success_cnt + 1))
	fi

	RCMsg "Checking Generic Kernel Storage Engine" \
	[ "x`/sbin/lsmod | /bin/grep target_core_mod | /usr/bin/awk '{ if ($1 == "target_core_mod") print $1 }'`" != "x" ]
	if [ $? -eq 0 ]; then
		success_cnt=$(($success_cnt + 1))
	fi

	RCMsg "Checking iSCSI Target Core Stack" \
	[ "x`/sbin/lsmod | /bin/grep iscsi_target_mod | /usr/bin/awk '{ if ($1 == "iscsi_target_mod") print $1 }'`" != "x" ]
	if [ $? -eq 0 ]; then
		success_cnt=$(($success_cnt + 1))
	fi

	return $success_cnt
}

lio_insmod()
{
	if [ ! -d /var/crash ]; then
		/bin/mkdir /var/crash
	fi

	if [ ! -d /var/lock/subsys ]; then
		/bin/mkdir -p /var/lock/subsys
	fi

	if [ ! -d /var/target/pr ]; then
		/bin/mkdir -p /var/target/pr
	fi

	if [ ! -d /var/target/alua ]; then
		/bin/mkdir -p /var/target/alua
	fi

	if [ ! -d /sys/kernel/config ]; then
		/sbin/insmod /lib/modules/configfs.ko
		/bin/mount -t configfs configfs /sys/kernel/config
		RCMsg "Mounting configfs on /sys/kernel/config" [ $? -eq 0 ]
	fi

        if [ -f /lib/modules/iscsi_extent_pool.ko ] ; then
		if [ "x`/sbin/lsmod | /bin/grep iscsi_extent_pool | /usr/bin/awk '{ if ($1 == "iscsi_extent_pool") print $1 }'`" = "x" ]; then
			RCMsg "Loading Generic Kernel Extent Pool Engine" \
			/sbin/insmod /lib/modules/iscsi_extent_pool.ko
		fi
	fi

	if [ "x`/sbin/lsmod | /bin/grep target_core_mod | /usr/bin/awk '{ if ($1 == "target_core_mod") print $1 }'`" = "x" ]; then
		RCMsg "Loading Generic Kernel Storage Engine" \
		/sbin/insmod /lib/modules/target_core_mod.ko
	fi

	if [ "x`/sbin/lsmod | /bin/grep iscsi_target_mod | /usr/bin/awk '{ if ($1 == "iscsi_target_mod") print $1 }'`" = "x" ]; then
		RCMsg "Loading iSCSI Target Core Stack" \
		/sbin/insmod /lib/modules/iscsi_target_mod.ko
	fi
}

lio_start()
{
	while ( ! iscsi_trash_lock_stat ) ; do
		sleep 1;
	done

	RCMsg "Running lunbackup garbage collection" \
	/usr/syno/bin/synolunbkp --gc

	local enabled_count=`/usr/syno/bin/iscsitrgtool -enabled_count`
	local trash_count=`/usr/syno/bin/iscsitrgtool -trashcount`
	if [ $enabled_count -eq 0 -a $trash_count -eq 0 ]; then
		RCMsg "No iSCSI targets enabled, skip it."
		RCMsg "No iSCSI VAAI lun in trash can, skip it."
		return $LSB_SUCCESS
	fi

	local vaai_support=`/bin/get_key_value /etc.defaults/synoinfo.conf support_vaai`
	if [ "x${vaai_support}" = "xyes" ]; then
		/usr/syno/sbin/epck --startup-replay
	fi

	lio_insmod

	iscsitrg_lock_trap_unset

	# workaround for delayed synoboot
	(
		iscsitrg_lock_trap_set

		cnt=0
		platform=`get_key_value /etc.defaults/synoinfo.conf unique | cut -d'_' -f2`
		while [ $cnt -lt 20 ] && [ "$platform" = "x86" -o "$platform" = "bromolow" ] && [ ! -d /sys/block/synoboot ]
		do
			sleep 3
			cnt=$(($cnt + 1))
		done
		/usr/syno/bin/iscsitrgtool -startall > /dev/null 2>&1
		if [ -f /lib/modules/iscsi_extent_pool.ko -a $trash_count -ne 0 ]; then
			RCMsg "Running VAAI luns garbage collection"

			iscsi_trash_lock
			(
				iscsi_trash_lock_trap_set

				/usr/syno/bin/iscsitrgtool -trashclean > /dev/null 2>&1
			) &
		fi
	) &

	return $LSB_SUCCESS
}

lio_stop()
{
	RCMsg "Terminating all lun backup process" \
	/usr/bin/killall -TERM synolunbkp
	sleep 3

	if [ ! -d /sys/kernel/config/target ]; then
		return 0
	fi

	local stop_ret=`/usr/syno/bin/iscsitrgtool -stopall`
	if [ $stop_ret -lt 0 ]; then
		MsgFail "Stopping iSCSI LIO targets"
	else
		RCMsg "Unloading iSCSI Target Core Stack" /sbin/rmmod iscsi_target_mod
		RCMsg "Unloading Generic Kernel Storage Engine" /sbin/rmmod target_core_mod

		if [ -f /lib/modules/iscsi_extent_pool.ko ] ; then
			rmdir /sys/kernel/config/pool/EP_* 2> /dev/null
			RCMsg "Unloading Generic Kernel Extent Pool Engine" /sbin/rmmod iscsi_extent_pool
		fi

		/bin/umount -l /sys/kernel/config
		RCMsg "Unloading configfs on /sys/kernel/config" /sbin/rmmod configfs

		echo 1 > /proc/sys/vm/drop_caches
	fi

	#check the consistent between files and configs
	/usr/syno/bin/synocheckiscsitrg > /dev/null 2>&1

	return $LSB_SUCCESS
}

## iSCSI target rc generic interface
iscsitrg_start()
{
	if [ "x${ISCSITRGT_TYPE}" = "xlio" ]; then
		lio_start
	else
		iet_start
	fi
}

iscsitrg_stop()
{
	if [ "x${ISCSITRGT_TYPE}" = "xlio" ]; then
		lio_stop
	else
		iet_stop
	fi
}

iscsitrg_restart()
{
	if [ "x${ISCSITRGT_TYPE}" = "xlio" ]; then
		lio_stop
		sleep 1
		lio_start
	else
		iet_stop
		sleep 1
		iet_start
	fi
}

iscsitrg_status()
{
	if [ "x${ISCSITRGT_TYPE}" = "xlio" ]; then
		lio_status
	else
		iet_status
	fi

	local ret=$?
	if [ $ret -eq $check_module_cnt ]; then
		RCMsg "iSCSI Service: running."
		return $LSB_STAT_RUNNING
	elif [ $ret -eq 0 ]; then
		RCMsg "iSCSI Service: stopped."
		return $LSB_STAT_NOT_RUNNING
	else
		RCMsg "iSCSI Service: module error."
		return $LSB_STAT_MODULE_ERR
	fi
}

iscsitrg_insmod()
{
	if [ "x${ISCSITRGT_TYPE}" = "xlio" ]; then
		lio_insmod
	else
		iet_insmod
	fi
}

check_support()
{
	case "${ISCSITRGT_SUPPORT}" in
		[Yy][Ee][Ss])
			return $LSB_SUCCESS
			;;
		*)
			RCMsg "iSCSI target is not supported."
			return $LSB_ERR_CONFIGURED
			;;
	esac
}

iscsi_trash_unlock()
{
    /bin/rm "${ISCSITRASH_LOCK}"
}

iscsi_trash_lock_trap_set()
{
    trap 'iscsi_trash_unlock' INT TERM EXIT ABRT
}

iscsi_trash_lock()
{
    echo $$ > ${ISCSITRASH_LOCK}
}

iscsi_trash_lock_stat()
{
    if [ -f ${ISCSITRASH_LOCK} ] ; then
        MsgWarn "Clean-Trash is working?"
        return 1
    fi

    return 0
}

iscsitrg_unlock()
{
    /bin/rm "${ISCSITRGT_LOCK}"
}

iscsitrg_lock_trap_unset()
{
    trap - INT TERM EXIT ABRT
}

iscsitrg_lock_trap_set()
{
    trap 'iscsitrg_unlock' INT TERM EXIT ABRT
}

iscsitrg_lock()
{
    if [ -f ${ISCSITRGT_LOCK} ]; then
        MsgWarn "$0 is working?"
        return 0
    fi

    iscsitrg_lock_trap_set

    echo $$ > ${ISCSITRGT_LOCK}
    return 0
}

[ $# != 0 ] && ! check_support && exit $?

iscsitrg_lock || exit $?

case $1 in
	start)
		iscsitrg_start
		;;

	stop)
		iscsitrg_stop
		;;

	restart)
		iscsitrg_restart
		;;

	status)
		iscsitrg_status
		;;

	insmod|just_start_service)
		iscsitrg_insmod
		;;

	*)
		echo "Usage: `/usr/bin/basename $0` {start|stop|restart|status}"
		exit 1
esac

exit $?
